/** @file   smallwormalien.cpp
 * @brief   Implementation of SmallWormAlien - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#include "smallwormalien.h" // class's header file
#include "smallwormai.h"
#include "gameanims.h"
#include "www_map.h"
#include "animplayer.h"
#include "GfxManager.h"
#include "soundsamples.h"
#include "settings.h"
#include "redrawqueue.h"
#include "warglobals.h"
#include "AnimId.h"
using namespace eng2d;


namespace WeWantWar {


const int SmallWormAlien::NUM_OF_BULLETS = 10;
const float SmallWormAlien::BULLET_RANGE = 250;


// class constructor
SmallWormAlien::SmallWormAlien() :
  Alien( ),
  m_spdVec( 0, -1.0 ),
  m_reloadingDelay( 180 )
{
  ObjectID::Type oid = this->objectType();
  // Set the moving speed.
  m_spdVec.y( - Settings::floatObjProp(oid, "spd_walk:") );
  
  this->setCorrectAnimation( GameAnims::EWalk );
  this->boundingSphere( Settings::floatObjProp(oid, "bounding_sphere:") );
  this->setArmor( Settings::floatObjProp(oid, "armor:") );
  m_reloadingDelay = Settings::intObjProp(oid, "reload:" );
  
  // 4 collisionpoints
  this->setCollisionPoint( 0, Vec2D( 0,-20) );
  this->setCollisionPoint( 1, Vec2D( 0, 20) );
  this->setCollisionPoint( 2, Vec2D( -20, 0) );
  this->setCollisionPoint( 3, Vec2D( 20, 0) );

  // Create the ai-object.
  this->setController( new SmallWormAI( this ) );
}

// class destructor
SmallWormAlien::~SmallWormAlien()
{
}



/** Reimplemented update-method.
 */
void SmallWormAlien::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  if ( this->state() == GameObject::STATE_DYING ) {
    if ( m_animation.paused() ) {
      // The alientype 2 fires some bullets when it dies
      this->state( GameObject::STATE_KILLED );
      this->setCorrectAnimation( GameAnims::EKilled );
      this->hidden( true );
      this->attack();
      const Animation& explosionAnim =
          GameAnims::findAnimation( AnimId::KExplosionSmallWorm, 0 );
      AnimPlayer::spawn( explosionAnim, this->position(), 0 );
      this->makeSound( GameObject::SND_DIE );
    }
    return;
  }

  // If we're running the shooting animation, we do nothing more before it's
  // shown
  if ( this->getAnimID() == GameAnims::EShoot ) {
    if ( m_animation.paused() ) {
      // The shooting animation has ended. Launch the bullets and set default
      // animation back
      this->attack();
      m_pController->setCounter( -1 );
      this->setCorrectAnimation( GameAnims::EWalk );
    }

  } else {
    m_pController->update();

    // Do the movement
    Vec2D v( m_spdVec );
    v.rotate( this->angle() );
    if ( m_pController->forward() == 1 ) {
      this->move( v );
    }
    this->changeAngle( m_pController->turn() );

    // We are supposed to shoot. We launch the shooting animation.
    if ( m_pController->shoot() != 0 ) {
      if ( this->reloading() == false ) {
        this->setCorrectAnimation( GameAnims::EShoot );
        m_pController->m_shoot = 0;
        this->setCounter( Alien::RELOAD_COUNTER_INDEX, m_reloadingDelay );
      }
    }
  }
}



/** Makes sound
 */
void SmallWormAlien::makeSound( GameObject::SoundID id ) const
{
  if ( id == GameObject::SND_PAIN ) {
    Sound::playSample( SMP_ALIEN2_PAIN, false );
    
  } else if ( id == GameObject::SND_DIE ) {
    Sound::playSample( SMP_ALIEN2_DIE, false );
    
  }
}



/** Kills this alien.
 */
void SmallWormAlien::kill()
{
  if ( this->state() == STATE_KILLED || this->state() == STATE_DYING ) {
    return;
  }
  this->state( STATE_DYING );
  this->setCorrectAnimation( GameAnims::EDying );
}



/** Hit by bullet
 */
bool SmallWormAlien::hitByBullet( Bullet* pB )
{
  bool ret = Alien::hitByBullet( pB );
  if ( ret == true ) {
    if ( pB->iType != Bullet::EFlameThrower ) {
      ParticleBlood* pP = new ParticleBlood( pB->iPosition, pB->velocity(),
                                             7, Color(130,60,10) );
      WarGlobals::pPartManager->addSystem( pP );
    }
  }
  return ret;
}



/** Returns the type id of this alien.
 */
ObjectID::Type SmallWormAlien::objectType() const
{
  return ObjectID::TYPE_SMALLWORMALIEN;
}



/** Returns the redraw priority
 */
int SmallWormAlien::redrawPriority() const
{
  return RedrawQueue::PRI_BELOW_NORMAL;
}



/** Emits some bullets.
 */
void SmallWormAlien::attack()
{
  float angleStep = 256 / static_cast<float>( NUM_OF_BULLETS );
  float shootAngle = 0;
  for ( int i=0; i < NUM_OF_BULLETS; i++ ) {
    Bullet* pB = BulletTable::createBullet( this, m_position, Bullet::ERifle );
    Vec2D spdV( pB->velocity() );
    spdV.rotate( shootAngle );
    pB->setVelocity( spdV );
    pB->iRange = BULLET_RANGE;
    WarGlobals::pBulletManager->spawnBullet( pB );
    shootAngle += angleStep;
  }
  const Animation& lightAnim = GameAnims::findAnimation( AnimId::KSmallWormShootLight );
  AnimPlayer::spawn( lightAnim, this->position(), 0 );
}



/** Sets the requested animation
 */
void SmallWormAlien::setCorrectAnimation( int animID )
{
  const Animation& anim = GameAnims::findAnimation( AnimId::KSmallWormAlien, animID );
  this->setAnimation( anim, animID);
}

} // end of namespace

